{******************************************************************************
 ******************************************************************************
 *
 *
 *  EZ-KIT Lite resident monitor
 *
 *  This file is part of the ADSP 2181 EZ-KIT Lite resident monitor.
 *
 *  The monitor is loaded from the on-board EPROM through the BDMA
 *  interface during power up.
 *
 *  The complete source codes include the following files:
 *
 *  INTVEC  .DSP : interrupt vector table
 *  UART    .DSP : UART simulation routine
 *  MAIN    .DSP : main module
 *  SELFTEST.DSP : power up self test routine
 *  UTILITY .DSP : send a line of ASCII text through serial line
 *  COMMAND .DSP : command loop and command interpreter
 *  CODEC   .DSP : 1847 codec initialization and interrupt handling
 *
 *  TONE    .DAT : 32 samples of one complete cycle sine wave
 *  CONSTANT.K   : constants
 *
 *
 ******************************************************************************
 ******************************************************************************}





.module/ram     COMMAND;

.include        <constant.k>;

.entry          lastCmdOK;
.entry          irq1isr;
.entry          cmdProc;

.global         usrRan;
.global         intsVec;

.var/dm/ram     usrRan, temp;
.var/dm/ram     ok [9], bad [11], aliveCnt;
.var/pm/ram/abs=0x3fc0     intsVec [0x30];      { do not change this address }

{ scratchpad variables during DMA xfer }
.var/dm/ram     dma_seg_len,
                dma_seg_start,
                dma_seg_type,
                dma_data_access,
								exec_start_addr,
								dma_return_data[3];

.init ok: 'P', 'r', 'o', 'm', 'p', 't', '>', 0, 0;
.init bad: 'W','h','u','h',0x00,' ','I','d','e', 'e', 0;
.init aliveCnt: 0x2f;
.init usrRan: 0;        { 1 if user program is running. }






{******************************************************************************
 *
 *  This command parser accepts these commands:
 *
 *  1.  Beep
 *      received:$$$
 *      send    :ok[led count][alive called count][selftest hi][selftest lo]
 *
 *
 *      [led count] is the lower 8 bits of a counter incremented at 28800 rate.
 *      [alive called count] is incremented by one each time alive is called.
 *      [selftest hi][selftest lo] is the selftest result
 *      beep
 *
 *  2.  Alive inquiry
 *      received:$OK
 *      send    :ok[led count][alive called count][selftest hi][selftest lo]
 *
 *      [led count] is the lower 8 bits of a counter incremented at 28800 rate.
 *      [alive called count] is incremented by one each time alive is called.
 *      [selftest hi][selftest lo] is the selftest result
 *
 *  3.  upload DM
 *      received:$UD[hi start][lo start][hi len][lo len]
 *      send    :[hi][lo]...[hi sum][lo sum]$!
 *
 *  4.  upload PM
 *      received:$UP[hi start][lo start][hi len][lo len]
 *      send    :[hi][mi][lo]...[hi sum][mi sum][lo sum]$!
 *
 *  5.  download DM
 *      received:$DD[hi start][lo start][hi len][lo len][hi][lo]...
 *      send    :![hi sum][lo sum]
 *
 *  6.  download PM
 *      received:$DP[hi start][lo start][hi len][lo len][hi][mi][lo]...
 *      send    :![hi sum][mi sum][lo sum]
 *
 *  7.  Call subroutine
 *      received:$GO[hi address][loaddress]
 *      send    :![hi address][loaddress]
 *
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  :
 *  update :
 *  output :
 *  destroy:
 *  keep   :
 *  memory :
 *  calls  :
 *
 ******************************************************************************}
{-- wait for first $ --}

cmdProc:
        call checkpfs;
        idle;
        call get_char_ax1;

        ay1 = 0x24;
        none = ax1 - ay1;
        if eq jump cmdPrmpt;        { receive command prompt $ }

        none = pass ax1;
        if eq jump cmdProc;         { skip 0x00 }


{-- command not regconized --}
        dm (bad + 9) = ax1;         { for others, send back string: you type:? }
        i7 = ^bad;
        call sendLine;
        jump cmdProc;


cmdPrmpt:
{-- now get the next two characters (will time out.) --}
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }

        none = pass ar;
        if eq jump cmdProc;     { if both zero, discard; host clearing }
{-- now the first character is in upper 8 bits of ar and second in lower. --}



{******************************************************************************
 *
 *  this is the command dispatcher.
 *
 ******************************************************************************}

        m7 = 1;
        l7 = 0;

        ay0 = 0x4f4b;           { ASCII for 'OK' }
        none = ar - ay0;
        if eq jump alive;


        ay0 = 0x2424;           { ASCII for '$$' }
        none = ar - ay0;
        if eq jump beep;


        ay0 = 0x5544;           { ASCII for 'UD' }
        none = ar - ay0;
        if eq jump updm;



        ay0 = 0x5550;           { ASCII for 'UP' }
        none = ar - ay0;
        if eq jump uppm;


        ay0 = 0x4444;           { ASCII for 'DD' }
        none = ar - ay0;
        if eq jump dndm;


        ay0 = 0x4450;           { ASCII for 'DP' }
        none = ar - ay0;
        if eq jump dnpm;


        ay0 = 0x474f;           { ASCII for 'GO' }
        none = ar - ay0;
        if eq jump go;

        ay0 = 0x4558;           { ASCII for 'EX' }
        none = ar - ay0;
        if eq jump execute;

        ay0 = 0x5255;           { ASCII for 'RU' }
        none = ar - ay0;
        if eq jump run;


{-- command not regconized --}
        dm (bad + 9) = ar;
        i7 = ^bad;


lastCmdBad:
        ar = BEEP_1000_mS;          { beep }
        dm (codecBeep) = ar;
        ar = LEDRateBad;            { get fast LED blink rate value. }
        dm (led_cntr_top) = ar;     { set led counter reset value. }
        jump cmdProc;

checkpfs:
        { set up PF0-PF3 }
        ay1 = dm(PFTYPE);
        ax1 = 0xFFF0;
        ar  = ax1 and ay1;
        dm(PFTYPE) = ar;

        { bus settle ? }
        nop;
        nop;

        { read PF inputs }
        ay1  = dm(PFDATA);
        ax1  = 0x0007;
        ar   = ax1 and ay1;
        none = pass ar;
        if ne jump usepfs;
        rts;

usepfs:
        { coolness }
        dm(exec_start_addr) = ar;
        call run2;
        rts;

lastCmdOK:
        ar = LEDRateGood;           { get slow LED blink rate value. }
        dm (led_cntr_top) = ar;     { set led counter reset value. }
        jump cmdProc;





{******************************************************************************
 *
 *  received:$$$
 *
 *  send    :ok[led count][alive called count][selftest hi][selftest lo]
 *
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : none
 *  update : none
 *  output : none
 *  destroy: none
 *  keep   : none
 *  memory : none
 *  calls  : none
 *
 ******************************************************************************}
beep:
        ar = BEEP_0100_mS;              { enable codec to beep }
        dm (codecBeep) = ar;


{******************************************************************************
 *
 *  received:$OK
 *
 *  send    :ok[led count][alive called count][selftest hi][selftest lo]
 *
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : none
 *  update : none
 *  output : none
 *  destroy:
 *  keep   : none
 *  memory : none
 *  calls  : none
 *
 ******************************************************************************}
alive:
{-- byte 2 is led counter; byte 3 is alive called count --}
        ar = 0x6f6b;                { send ok }
        call out_int_ar;

        ax1 = dm (led_cntr);        { get led counter }
        call out_char_ax1;          { send lower 8 bits }

        ar = dm (aliveCnt);         { get and increment aliveCnt }
        ar = ar + 1;
        dm (aliveCnt) = ar;
        ax1 = ar;
        call out_char_ax1;          { send lower 8 bits }

        ar = dm (selfTstRst);       { get selftest result }
        call out_int_ar;
        ar = 0x0a0d;                { send line feed/carriage return }
        call out_int_ar;
        jump lastCmdOK;



{******************************************************************************
 *
 *  received:$UD[hi start][lo start][hi len][lo len]
 *
 *  send    :[hi][lo]...[hi sum][lo sum]$!
 *
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : none
 *  update : none
 *  output : none
 *  destroy: none
 *  keep   : none
 *  memory : none
 *  calls  : none
 *
 ******************************************************************************}
updm:
{   get start }
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }
        i7 = ar;

{   get len }
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }

{   the loop }
        af = pass 0;                { zero checksum }
        cntr = ar;
        do updm0 until ce;
        ar = dm (i7, m7);
        af = ar + af;               { compute checksum }
        call out_int_ar;            { send to host }
updm0:  nop;

        ar = pass af;               { send checksum }
        call out_int_ar;

        jump lastCmdOK;




{******************************************************************************
 *
 *  received:$UP[hi start][lo start][hi len][lo len]
 *
 *  send    :[hi][mi][lo]...[hi sum][mi sum][lo sum]$!
 *
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : none
 *  update : none
 *  output : none
 *  destroy: none
 *  keep   : none
 *  memory : none
 *  calls  : none
 *
 ******************************************************************************}
uppm:
{   get start }
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }
        i7 = ar;

{   get len }
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }

{   the loop }
        mr = 0;                     { zero checksum }
        cntr = ar;
        do uppm0 until ce;

        ay1 = pm (i7, m7);

        { compute checksum }
        sr1 = 0;                    { px in lower 8 sr0 }
        sr0 = px;
        si = ay1;
        sr = sr or lshift si by 8 (lo);    { or in upper 16 bits }
        ay0 = mr0;
        ar = sr0 + ay0;             { do a 24 bits add; hi:lo in mr1:mr0 }
        mr0 = ar;                   { number is in lower 24 bits of 32 bits }
        ay0 = mr1;
        ar = sr1 + ay0 + c;         { add with carry }
        mr1 = ar;

        ar = ay1;
        call out_int_ar;            { send upper 16 bits. }
        ax1 = px;
        call out_char_ax1;          { send lower 8 bits. }
uppm0:  nop;

{   compute and send 24 bits checksum }
{   these codes are also used by download pm }
        sr = lshift mr0 by 8 (lo);  { shirt upper 16 bits into sr1 }
        sr = sr or lshift mr1 by 8 (hi);  { shirt upper 16 bits into sr1 }
        ar = sr1;
        call out_int_ar;            { send upper 16 bits. }
        ax1 = mr0;                  { lower 8 bits still in mr0 }
        call out_char_ax1;          { and lower 8 bits. }

        jump lastCmdOK;


{******************************************************************************
 *
 *  received:$DD[hi start][lo start][hi len][lo len][hi][lo]...
 *
 *  send    :![hi sum][lo sum]
 *
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : none
 *  update : none
 *  output : none
 *  destroy: none
 *  keep   : none
 *  memory : none
 *  calls  : none
 *
 ******************************************************************************}
dndm:
{   get start }
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }
        i7 = ar;

{   get len }
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }

{   the loop }
        af = pass 0;
        cntr = ar;
        do dndm0 until ce;
        call get_int_ar_to;
        if lt jump dndmBad;         { check time out: jump if timeout }
        dm (i7, m7) = ar;
dndm0:  af = ar + af;               { compute checksum }

{   send checksum }
        ar = pass af;               { send checksum }
        call out_int_ar;

        jump lastCmdOK;


dndmBad:
{   time out, clean up counter, loop, and PC stacks. }
{   These codes are also used by download pm }
      { pop cntr; }         { no need to pop counter because it's empty. }
        pop pc;                     { popping relavent stacks since we }
        pop loop;                   { are jumping out of do loop. }
        jump lastCmdBad;



{******************************************************************************
 *
 *  received:$DP[hi start][lo start][hi len][lo len][hi][mi][lo]...
 *
 *  send    :![hi sum][mi sum][lo sum]
 *
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : none
 *  update : none
 *  output : none
 *  destroy: none
 *  keep   : none
 *  memory : none
 *  calls  : none
 *
 ******************************************************************************}
dnpm:
{   get start }
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }
        i7 = ar;

{   get len }
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }

{   the loop }
        mr = 0;
        cntr = ar;
        do dnpm0 until ce;

        call get_int_ar_to;
        if lt jump dndmBad;         { check time out: share download dm codes }
        ay1 = ar;                   { save upper 16 bits in ay1 }
        call get_char_ax1_to;       { get lowest 8 bits. }
        if lt jump dndmBad;         { check time out: share download dm codes }
        px = ax1;                   { load px }

        { compute checksum }
        sr1 = 0;                    { px in lower 8 sr0 }
        sr0 = px;
        si = ay1;
        sr = sr or lshift si by 8 (lo);    { or in upper 16 bits }
        ay0 = mr0;
        ar = sr0 + ay0;             { do a 24 bits add; hi:lo in mr1:mr0 }
        mr0 = ar;                   { number is in lower 24 bits of 32 bits }
        ay0 = mr1;
        ar = sr1 + ay0 + c;         { add with carry }
        mr1 = ar;

dnpm0:  pm (i7, m7) = ay1;          { save to pm }

{   compute and send 24 bits checksum }
        jump uppm0;                 { share upload pm code }





{******************************************************************************
 *
 *  received:$GO[hi address][loaddress]
 *
 *  send    :![hi address][loaddress]
 *
 *  user program to jump to 0x3fff to continue monitor program.
 *
 *
 *  before running user program:
 *  1.  dis timer
 *  2.  enable IRQ1
 *  3.  mask all ints
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : none
 *  update : none
 *  output : none
 *  destroy: none
 *  keep   : none
 *  memory : none
 *  calls  : none
 *
 ******************************************************************************}
go:
{   get address }
        call get_int_ar_to;
        if lt jump lastCmdBad;      { check time out }

run_program:
        i7 = ar;
        call out_int_ar;            { echo address back to host }
go1:    ar = dm (flag_tx_ready);    { wait until last transmit is done. }
        none = pass ar;
        if eq jump go1;

        ar = 1;                     { set user program running flag }
        dm (usrRan) = ar;

        dis timer;                  {   disable timer (will be enable by IRQ1 }
        imask = 0;


        ax0 = 0;
        dm (System_Control_Reg) = ax0;  {   shut down sport 0 }
        dm (SPORT0_Autobuf) = ax0;      {   turn off auto-buffering }

        ay1 = i7;                   { copy 0x3fc0..ef 0x00..2f }
        i4 = 0;                     { PC will download 0..2f to 3fc0..ef }
        l4 = 0;
        m4 = 0;
        i7 = ^intsVec;
        m7 = 1;
        cntr = %intsVec;
        do usrVec until ce;
        ax0 = pm (i4, m4);          { current int vec to be saved }
        ax1 = px;
        ay0 = pm (i7, m4);          { downloaded int vec to move to low mem }
        {   px implicit }
        pm (i4, m7) = ay0;
        px = ax1;
usrVec: pm (i7, m7) = ax0;


        i7 = ay1;
{   call USER codes }
        call (i7);
{   returned from USER codes }

        dis ints;
        dis timer;
        ifc = b#00000011111111;         { clear pending interrupt }
        nop;

{   pop stacks; in case user program did not empty stacks }
        pop sts;       pop sts;       pop sts;       pop sts;
        pop cntr;      pop cntr;      pop cntr;      pop cntr;
        pop pc;        pop pc;        pop pc;        pop pc;
        pop loop;      pop loop;      pop loop;      pop loop;


        ar = 0;                     { clear user program running flag }
        dm (usrRan) = ar;

{   restore just in case }
        l7 = 0;
        m7 = 1;                 { restore for memory transfer }


        i4 = 0;                     { restore saved int vec. }
        l4 = 0;
        i7 = ^intsVec;
        cntr = %intsVec;
        do usrVec1 until ce;
        ar = pm (i7, m7);          { downloaded int vec to move to low mem }
usrVec1:pm (i4, m7) = ar;          {   px implicit }

        icntl = b#00010;
        mstat = b#1000000;
        imask = b#0000000001;

        ar = 1;
        dm (codecBeep) = ar;

        ena ints;

        call codecInit;

				call init_uart;             { intialize uart routine }
				call turn_rx_on;            { turn on uart }

				ar = BEEP_0100_mS;          { enable codec to beep }
				dm (codecBeep) = ar;

				jump lastCmdOK;


{
	++++++++++++++++++++++++++++++++++++++++++..........................
	+
	+   RUN, run, you better run.   .  . . ..... . .   .     .    .  .  . . ...
	+
	+           [byte][byte][byte][     byte][          byte][         byte]
	+  received:[   R][   U][   0][  prg_num][ start addr hi][start addr lo]
	+  val     :[0x52][0x55][0x00][0x01-0x3f][     0x00-0x3f][    0x00-0xff]
	+           [ command  ][ program number][ start address               ]
	+
	+   Run functions exactly like execute except that after
	+   loading the program, it executes it.  It begins its
	+   execution in program memory at the start address it reads
	+   from the console port.  Usually, this is 0x0000.																																																															imagine that...it begins its execution at address 0.
	+
	++++++++++++++...............................
}

run:
        { get command line parameters ----------------------------------- }
        call get_int_ar_to;             { read start page                 }
        dm(exec_start_addr) = ar;
        {if lt jump lastCmdBad;}          { check time out                  }
run2:
        call n_out_int_ar;              { echo addr back to host          }

        { load program }
        call load_prg;                                                                  { load that program                                                             }

        { set start addr to 0 }
        ar  = 0x0000;
        call run_program;                                                               { run that program                                                              }

        jump lastCmdOK;

{******************************************************************************
 *
 *   Execute a DSP Program .... .... ... ... .. .. . .  .   .    .     .
 *
 *
 *  You have to imagine it asking itself,
 *  "Where would I be, if I would like to reproduce myself?"
 *
 *	received:[   E][   X][   0][prg_num]
 *  type:        [byte][byte][byte][   byte]
 *      val:     [0x45][0x58][0x00][  hex  ]
 *
 *  e.g.	   At the command prompt, sending
 *           [   $][   E][   X][   n][   n]
 *           [0x24][0x45][0x48][0x00][0x01] will load and execute
 *					 program number 1.
 *
 *	Note that commands do not have to be followed by carriage returns.
 *
 *      ** *  *   *    Return Values    *   *  * **
 *  As this program loads segments from the DMA interface into the
 *  DSP's internal SRAM, it echoes the segment headers to the screen
 *  in the following format.
 *
 *  Example: Time Distortion Rev 1.1 is loaded from 0x04000.
 *
 *  Initial:  0x00 0x01   ( program number        )   [ 1 - 63 ]
 *            0x00 0x00   ( BDMA BEAD             )
 *            0x01 0x03   ( BDMA Ctrl             )   [ 16KB page number ][ DMA-MODE ]
 *  Segment:  0x01 0x78   ( Segment Length        )
 *  Header :  0x00 0x00   ( Segment Start Addr    )
 *            0x00 0x02   ( Segment Type ) [ 1 = data, 2 = program, 0 = end ]
 *            etc .
 *            etc .
 *            enough.
 *            already.
 *
 *    prg_num selects the 16KB page from which to begin loading
 *    It refers to the lowest
 *    three bits of BMPAGE in the BDMA Control Register
 *
 *    e.g. For a 128KB EPROM, prg num ranges from 0-7
 *    	   For a 1MB   EPROM, prg num ranges from 0-63
 *
 *  calls  : get_int_ar_to, n_out_int_ar, dma read funcs
 *
 ******************************************************************************}

execute:

        { get command line parameters ----------------------------------- }
        call get_int_ar_to;             { read start page                 }
        dm(exec_start_addr) = ar;

{        if lt jump lastCmdBad;    }      { check time out                  }
        call n_out_int_ar;              { echo addr back to host          }

        ax0 = 0x4400;
        ay0 = 0x0045;
        ar  = ax0 or ay0;
        call n_out_int_ar;

        ax0 = 0x4600;
        ay0 = 0x0047;
        ar  = ax0 + ay0;
        call n_out_int_ar;

        call load_prg;
        jump lastCmdOK;

{ Note, this function has dm(exec_start_addr) set already when it.s called }

load_prg:
        { initialize DMA ------------------------------------------------ }

				ax1 = 0x0000;                   { set external address ---------- }
				dm(BDMA_BEAD) = ax1;
				ar = ax1;                       { echo to host                    }
				call n_out_int_ar;

				ar = dm(exec_start_addr);       { select 16K page (data) -------- }
				sr = lshift ar by 8 (lo);
				ay0 = 0x0003;                   { enable run during BDMA, load from BM, DM/LSB }
				ar = sr0 or ay0;
				dm(dma_data_access) = ar;
				call n_out_int_ar;

				ax1 = dm(dma_data_access);      { load data into temp DM, 				}
				dm(BDMA_BDMA_Ctrl) = ax1;				{ then move it where we want it   }

read_segment_header:

				{ read segment length ------------------------------------------- }
				call read_two_dma_bytes;        { returns word in ax1;            }
				dm(dma_seg_len)   = ax1;
				ar   = ax1;
				call n_out_int_ar;              { echo length to host             }

				{ read segment start -------------------------------------------- }
				call read_two_dma_bytes;        { returns word in ax1             }
				dm(dma_seg_start) = ax1;
				ar   = ax1;
				call n_out_int_ar;              { echo segment start to host      }

				{ read segment type --------------------------------------------- }
				call read_dma_byte;             { returns byte in ax1;            }
				dm(dma_seg_type)  = ax1;
				ar = ax1;
				call n_out_int_ar;              { echo segment type to host       }

				{ pick which type of segment and load it ------------------------ }
				ax1 = dm(dma_seg_type);
				ay0 = 0;{ end loading } none = ax1 - ay0;if eq jump done_with_exec;
				ay0 = 1;{ data seg    } none = ax1 - ay0;if eq jump do_data_seg;
				ay0 = 2;{ prg seg     } none = ax1 - ay0;if eq jump do_prog_seg;

				jump done_with_exec;            { other?  bad!  exit.             }


{ load a data segment --------------------------------------------------- }
do_data_seg:

				cntr = dm(dma_seg_len);         { prepare countdown               }
				m1   = 1;                       { prepare pointer                 }
				l1   = 0;
				i1   = dm(dma_seg_start);

				do store_data_word until ce;
				call read_two_dma_bytes    ;    { returns data word in ax1        }
store_data_word:
				dm(i1,m1) = ax1            ;

				jump read_segment_header   ;


{ load a program segment ------------------------------------------------ }
do_prog_seg:
				cntr = dm(dma_seg_len);         { prepare countdown               }
				m4   = 1;                       { prepare pointer                 }
				l4   = 0;
				i4   = dm(dma_seg_start);

				do store_prog_word until ce;
                                ar   = i4;
                                call n_out_int_ar;              { echo addr back to host          }
                                                                        
				call read_three_dma_bytes  ;    { returns prog word in ax1 and px }

store_prog_word:
                                pm(i4,m4) = ax1            ;

				jump read_segment_header   ;

{ FINI! ----------------------------------------------------------------- }
done_with_exec:
				rts;


{ READ A DMA BYTE ------------------------------------------------------- }
read_dma_byte:

				i2 = ^dma_return_data;
				dm(BDMA_BIAD) = i2;             { return byte to dma_return_data  }

				ax1 = 1;                        { init transfer                   }
				dm(BDMA_BWCOUNT) = ax1;
				call wait_dma_complete;         { wait until transfer is complete }

				ax1 = dm(dma_return_data);      { return byte                     }
				rts;

{ READ TWO DMA BYTES ---------------------------------------------------- }
read_two_dma_bytes:

				i2 = ^dma_return_data;
				dm(BDMA_BIAD) = i2;             { return word to dma_return_data  }

				ax1 = 2;
				dm(BDMA_BWCOUNT) = ax1;         { init transfer                   }

				call wait_dma_complete;         { wait until transfer is complete }
				call stuff_bytes_together;      { stuff bytes together            }

				ax1 = ar;                       { return word                     }
				rts;

{ READ THREE DMA BYTES -------------------------------------------------- }
read_three_dma_bytes:

        i2 = ^dma_return_data;
        dm(BDMA_BIAD) = i2;             { return bytes to dma_return_data }

        ax1 = 3;
        dm(BDMA_BWCOUNT) = ax1;

        call wait_dma_complete;         { wait until transfer is complete }
        call stuff_bytes_together;      { stuff bytes together            }

        ax1 = dm(i0,m0);                { return program word             }
        px  = ax1;
        ax1 = ar;                       
        rts;

{ - - -- -- --- --- ---- ---- WAIT_DMA_COMPLETE ---- ---- --- --- -- -- - }
wait_dma_complete:      								{ wait until transfer is complete }
        ar = dm(BDMA_BWCOUNT);
        none = pass ar;
        if ne jump wait_dma_complete;
				rts;

{ - - -- -- --- --- ---- ---- STUFF_BYTES_TOGETHER - ---- --- --- -- -- - }
stuff_bytes_together:
        i0  = ^dma_return_data;         { stuff bytes together                                          }
        m0  = 1;
        l0  = 0;
        ar  = dm(i0,m0);
        sr  = lshift ar by 8 (lo);
        ay1 = dm(i0,m0);
        ar  = sr0 or ay1;                                                               { return the lower two bytes of   }
        rts;                                                                                                            { two consecutive bytes together  }

{******************************************************************************
 *  A high to low transition on flag_in signifies the start bit; it also
 *  triggers IRQ1 ISR which then turn on timer if the timer is off.  This is
 *  at most 1/3 bit period too late but we should still catch the byte.
 *
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : none
 *  update : none
 *  output : none
 *  destroy: none
 *  keep   : none
 *  memory : none
 *  calls  : none
 *
 ******************************************************************************}
irq1isr:
        pop sts;
        ena timer;              { start timer now }
        rts;                    { note rts }



.endmod;
