{******************************************************************************
 ******************************************************************************
 *
 *
 *  EZ-KIT Lite resident monitor
 *
 *  This file is part of the ADSP 2181 EZ-KIT Lite resident monitor.
 *
 *  The monitor is loaded from the on-board EPROM through the BDMA
 *  interface during power up.
 *
 *  The complete source codes include the following files:
 *
 *  INTVEC  .DSP : interrupt vector table
 *  UART    .DSP : UART simulation routine
 *  MAIN    .DSP : main module
 *  SELFTEST.DSP : power up self test routine
 *  UTILITY .DSP : send a line of ASCII text through serial line
 *  COMMAND .DSP : command loop and command interpreter
 *  CODEC   .DSP : 1847 codec initialization and interrupt handling
 *
 *  TONE    .DAT : 32 samples of one complete cycle sine wave
 *  CONSTANT.K   : constants
 *
 *
 ******************************************************************************
 ******************************************************************************}

.module/ram     CODEC;


.const  L_CTRL_REG_VALUE=   0xc002;         /*  EZ-KIT values */
.const  R_CTRL_REG_VALUE=   0xc102;

.include    <constant.k>;

.entry      next_cmd;           { CODEC initialization commands }
.entry      input_samples;      { CODEC input interrupt }
.entry      codecInit;          { CODEC initialization }
.entry      codecStop;          { CODEC stop: irq and auto-buffering }
.global     codecBeep;          { CODEC beep }

.global     stat_flag;          { CODEC init status }
.global     rx_buf, tx_buf;


{   DO CHANGE THE FOLLOWING BLOCK OF DECLARATIONS.
    THIS IS SO THAT THEY GO TO A KNOWN LOCATION IN DM.
}

.global flag_rx_no_word;    { make available to codec.dsp.  demo prg checks.}
.global flag_tx_ready;      { make available to codec.dsp.  demo prg checks.}
.global flag_rx_ready;      { make available to codec.dsp.  demo prg checks.}

{   All fix location system variables are here. }

.var/dm/ram/abs=0x3e00   rx_buf[3];      { Status + L data + R data }
.var/dm/ram/abs=0x3e04   tx_buf[3];      { Cmd + L data + R data    }
.var/dm/ram/abs=0x3e08   ptr_to_get;     { pointer to routines, see .init }
.var/dm/ram/abs=0x3e09   flag_rx_no_word; { demo program check for byte from host:
                                            0=char ready, 1=not ready }
.var/dm/ram/abs=0x3e0a   ptr_to_out;     { pointer to routines, see .init }
.var/dm/ram/abs=0x3e0b   flag_tx_ready;  { demo program check to see if send to
                                        host is done: 0: transmitting, 1: done}
.var/dm/ram/abs=0x3e0c   flag_rx_ready;  { demo program check to see if
                                           currently: 0: receiving, 1: done}
.var/dm/ram/abs=0x3e0d   ptr_to_codec_ini;  { pointer to routines, see .init }
.var/dm/ram/abs=0x3e10   init_cmds[13];


.init ptr_to_get:       ^get_char_ax1_to;
.init ptr_to_out:       ^out_char_ax1;
.init ptr_to_codec_ini: ^codecInit;

{   END: DO CHANGE THE PRECEDING BLOCK OF DECLARATIONS. }

.var/dm/ram                     stat_flag, codecBeep;
.var/dm/ram                     tone [32];

.init tx_buf:   0xc000, 0x0000, 0x0000; /* Initially set MCE        */
.init codecBeep: 1;
{   32 points sine wave. }
.init tone: <tone.dat>;

{   These values are also checked in self-test;  they must agree. }
.init init_cmds:
        L_CTRL_REG_VALUE,
                    {   EZ-KIT: 0xc00f
                        Left input control reg
                        b7-6: 0=left line 1
                              1=left aux 1
                              2=left line 2
                              3=left line 1 post-mixed loopback
                        b5-4: res
                        b3-0: left input gain x 1.5 dB
                    }
        R_CTRL_REG_VALUE,
                    {   EZ-KIT: 0xc10f
                        Right input control reg
                        b7-6: 0=right line 1
                              1=right aux 1
                              2=right line 2
                              3=right line 1 post-mixed loopback
                        b5-4: res
                        b3-0: right input gain x 1.5 dB
                    }
        0xc288,     {
                        left aux 1 control reg
                        b7  : 1=left aux 1 mute
                        b6-5: res
                        b4-0: gain/atten x 1.5, 08= 0dB, 00= 12dB
                    }
        0xc388,     {
                        right aux 1 control reg
                        b7  : 1=right aux 1 mute
                        b6-5: res
                        b4-0: gain/atten x 1.5, 08= 0dB, 00= 12dB
                    }
        0xc488,     {
                        left aux 2 control reg
                        b7  : 1=left aux 2 mute
                        b6-5: res
                        b4-0: gain/atten x 1.5, 08= 0dB, 00= 12dB
                    }
        0xc588,     {
                        right aux 2 control reg
                        b7  : 1=right aux 2 mute
                        b6-5: res
                        b4-0: gain/atten x 1.5, 08= 0dB, 00= 12dB
                    }
        0xc680,     {
                        left DAC control reg
                        b7  : 1=left DAC mute
                        b6  : res
                        b5-0: attenuation x 1.5 dB
                    }
        0xc780,     {
                        right DAC control reg
                        b7  : 1=right DAC mute
                        b6  : res
                        b5-0: attenuation x 1.5 dB
                    }
        0xc85b,     {
                        data format register
                        b7  : res
                        b5-6: 0=8-bit unsigned linear PCM
                              1=8-bit u-law companded
                              2=16-bit signed linear PCM
                              3=8-bit A-law companded
                        b4  : 0=mono, 1=stereo
                        b0-3: 0=  8.
                              1=  5.5125
                              2= 16.
                              3= 11.025
                              4= 27.42857
                              5= 18.9
                              6= 32.
                              7= 22.05
                              8=   .
                              9= 37.8
                              a=   .
                              b= 44.1
                              c= 48.
                              d= 33.075
                              e=  9.6
                              f=  6.615
                       (b0) : 0=XTAL1 24.576 MHz; 1=XTAL2 16.9344 MHz
                    }
        0xc909,     {
                        interface configuration reg
                        b7-4: res
                        b3  : 1=autocalibrate
                        b2-1: res
                        b0  : 1=playback enabled
                    }
        0xca00,     {
                        pin control reg
                        b7  : logic state of pin XCTL1
                        b6  : logic state of pin XCTL0
                        b5  : master - 1=tri-state CLKOUT
                              slave  - x=tri-state CLKOUT
                        b4-0: res
                    }
        0xcc40,     {
                        miscellaneous information reg
                        b7  : 1=16 slots per frame, 0=32 slots per frame
                        b6  : 1=2-wire system, 0=1-wire system
                        b5-0: res
                    }
        0xcd00;     {
                        digital mix control reg
                        b7-2: attenuation x 1.5 dB
                        b1  : res
                        b0  : 1=digital mix enabled
                    }


{******************************************************************************
 *
 *  CODEC stops IRQ and auto-buffering; codecInit require afterward.
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : ar, ax0, ay0
 *  update : imask
 *  output : none
 *  destroy: none
 *  keep   : none
 *  memory : none
 *  calls  : none
 *
 ******************************************************************************}
codecStop:
        imask = 0;

        ax0 = 0;
        dm (SPORT0_Autobuf) = ax0;
        dm (SPORT0_Control_Reg) = ax0;

        ax0 = dm (System_Control_Reg);
        ay0 = b#1110111111111111;
            {   +-/!||+-----/+-/- | program memory wait states
                |  !|||           | 0
                |  !|||           |
                |  !||+---------- 0
                |  !||            0
                |  !||            0
                |  !||            0
                |  !||            0
                |  !||            0
                |  !||            0
                |  !|+----------- SPORT1 1=serial port, 0=FI, FO, IRQ0, IRQ1,..
                |  !+------------ SPORT1 1=enabled, 0=disabled
                |  +============= SPORT0 1=enabled, 0=disabled
                +---------------- 0
                                  0
                                  0
            }
        ar = ax0 and ay0;
        dm (System_Control_Reg) = ar;


        rts;



{******************************************************************************
 *
 *  CODEC initialization routine.
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  : none
 *  update : none
 *  output : none
 *  destroy: ar, af, ax0, ay0, i5, m5, l5, i6, l6, i7, l7,
 *  keep   : none
 *  memory :
 *      variables:
 *           dm(stat_flag)
 *           dm(tx_buf)
 *           dm(rx_buf)
 *           dm(init_cmds)
 *
 *      memory mapped control registers:
 *           dm (SPORT0_Autobuf)
 *           dm (SPORT0_RFSDIV)
 *           dm (SPORT0_SCLKDIV)
 *           dm (SPORT0_Control_Reg)
 *           dm (SPORT0_TX_Channels0)
 *           dm (SPORT0_TX_Channels1)
 *           dm (SPORT0_RX_Channels0)
 *           dm (SPORT0_RX_Channels1)
 *           dm (System_Control_Reg)
 *  calls  : none
 *
 ******************************************************************************}
codecInit:
        {   shut down sport 0 during re-initialization }
        call codecStop;


        ifc = b#00000011111111;         { clear pending interrupt }
        nop;


        i5 = ^rx_buf;
        l5 = %rx_buf;
        i6 = ^tx_buf;
        l6 = %tx_buf;
        i7 = ^init_cmds;
        l7 = %init_cmds;

        m5 = 1;




{================== S E R I A L   P O R T   #0   S T U F F ==================}
        ax0 = b#0000110011010111;   dm (SPORT0_Autobuf) = ax0;
            {   |||!|-/!/|-/|/|+- receive autobuffering 0=off, 1=on
                |||!|  ! |  | +-- transmit autobuffering 0=off, 1=on
                |||!|  ! |  +---- | receive m?
                |||!|  ! |        | m5
                |||!|  ! +------- ! receive i?
                |||!|  !          ! i5
                |||!|  !          !
                |||!|  +========= | transmit m?
                |||!|             | m5
                |||!+------------ ! transmit i?
                |||!              ! i6
                |||!              !
                |||+============= | BIASRND MAC biased rounding control bit
                ||+-------------- 0
                |+--------------- | CLKODIS CLKOUT disable control bit
                +---------------- 0
            }

        ax0 = 0;    dm (SPORT0_RFSDIV) = ax0;
            {   RFSDIV = SCLK Hz/RFS Hz - 1 }
        ax0 = 0;    dm (SPORT0_SCLKDIV) = ax0;
            {   SCLK = CLKOUT / (2  (SCLKDIV + 1) }
        ax0 = b#1000011000001111;   dm (SPORT0_Control_Reg) = ax0;
            {   multichannel
                ||+--/|!||+/+---/ | number of bit per word - 1
                |||   |!|||       | = 15
                |||   |!|||       |
                |||   |!|||       |
                |||   |!||+====== ! 0=right just, 0-fill; 1=right just, signed
                |||   |!||        ! 2=compand u-law; 3=compand A-law
                |||   |!|+------- receive framing logic 0=pos, 1=neg
                |||   |!+-------- transmit data valid logic 0=pos, 1=neg
                |||   |+========= RFS 0=ext, 1=int
                |||   +---------- multichannel length 0=24, 1=32 words
                ||+-------------- | frame sync to occur this number of clock
                ||                | cycle before first bit
                ||                |
                ||                |
                |+--------------- ISCLK 0=ext, 1=int
                +---------------- multichannel 0=disable, 1=enable
            }
            {   non-multichannel
                |||!|||!|||!+---/ | number of bit per word - 1
                |||!|||!|||!      | = 15
                |||!|||!|||!      |
                |||!|||!|||!      |
                |||!|||!|||+===== ! 0=right just, 0-fill; 1=right just, signed
                |||!|||!||+------ ! 2=compand u-law; 3=compand A-law
                |||!|||!|+------- receive framing logic 0=pos, 1=neg
                |||!|||!+-------- transmit framing logic 0=pos, 1=neg
                |||!|||+========= RFS 0=ext, 1=int
                |||!||+---------- TFS 0=ext, 1=int
                |||!|+----------- TFS width 0=FS before data, 1=FS in sync
                |||!+------------ TFS 0=no, 1=required
                |||+============= RFS width 0=FS before data, 1=FS in sync
                ||+-------------- RFS 0=no, 1=required
                |+--------------- ISCLK 0=ext, 1=int
                +---------------- multichannel 0=disable, 1=enable
            }


        ax0 = b#0000000000000111;   dm (SPORT0_TX_Channels0) = ax0;
            {   ^15          00^   transmit word enables: channel # == bit # }
        ax0 = b#0000000000000111;   dm (SPORT0_TX_Channels1) = ax0;
            {   ^31          16^   transmit word enables: channel # == bit # }
        ax0 = b#0000000000000111;   dm (SPORT0_RX_Channels0) = ax0;
            {   ^15          00^   receive word enables: channel # == bit # }
        ax0 = b#0000000000000111;   dm (SPORT0_RX_Channels1) = ax0;
            {   ^31          16^   receive word enables: channel # == bit # }


{============== S Y S T E M   A N D   M E M O R Y   S T U F F ==============}
        ax0 = b#0001000000000000;
            {   +-/!||+-----/+-/- | program memory wait states
                |  !|||           | 0
                |  !|||           |
                |  !||+---------- 0
                |  !||            0
                |  !||            0
                |  !||            0
                |  !||            0
                |  !||            0
                |  !||            0
                |  !|+----------- SPORT1 1=serial port, 0=FI, FO, IRQ0, IRQ1,..
                |  !+------------ SPORT1 1=enabled, 0=disabled
                |  +============= SPORT0 1=enabled, 0=disabled
                +---------------- 0
                                  0
                                  0
            }
        ay0 = dm (System_Control_Reg);
        ar = ax0 or ay0;
        dm (System_Control_Reg) = ar;





{******************************************************************************
 *
 *  ADSP 1847 Codec intialization
 *
 ******************************************************************************}

        {   enable transmit interrupt }
        imask = b#0001000000;
            {     |||||||||+ | timer
                  ||||||||+- | SPORT1 rec or IRQ0
                  |||||||+-- | SPORT1 trx or IRQ1
                  ||||||+--- | BDMA
                  |||||+---- | IRQE
                  ||||+----- | SPORT0 rec
                  |||+------ | SPORT0 trx
                  ||+------- | IRQL0
                  |+-------- | IRQL1
                  +--------- | IRQ2
            }


        {   set flag, disable tx ints }
        ax0 = 0;
        dm(stat_flag) = ax0;

{   The following codes make sure the address register wrap around is
    in sync with the frame and verifies the codec is responding correctly. }
again:
        ax0 = 0;                    { tx 0 to left and right }
        dm (tx_buf + 1) = ax0;
        dm (tx_buf + 2) = ax0;
        ax0 = dm (init_cmds + 11);  { codec to switch to 32 slot per frame }
        dm (tx_buf) = ax0;

        ax0 = dm (i6, m5);          { start interrupt }
        tx0 = ax0;

        idle;

        ax0 = 0x8dac;               { write ac to Digital mix control }
        dm (tx_buf) = ax0;

        idle;
        ax0 = 0x2d00;               { readback digital mix control }
        dm (tx_buf) = ax0;
        idle;
        idle;                       { for 32 slot, readback is in next frame }
        ax0 = dm (rx_buf);
        ay0 = 0x2dac;
        none = ax0 - ay0;
        if ne jump again;           { if not reading back, not in sync }


        ax0 = 0x8d50;               { write 50 to Digital mix control }
        dm (tx_buf) = ax0;

        idle;
        ax0 = 0x2d00;               { readback digital mix control }
        dm (tx_buf) = ax0;
        idle;
        idle;                       { for 32 slot, readback is in next frame }
        ax0 = dm (rx_buf);
        ay0 = 0x2d50;
        none = ax0 - ay0;
        if ne jump again;           { if not reading back, not in sync }


        {   clear flag }
        ax0 = 1;
        dm(stat_flag) = ax0;

        ax0 = dm (init_cmds + 12);  { codec to switch to 32 slot per frame }
        dm (tx_buf) = ax0;


check_init:
        idle;
        ax0 = dm (stat_flag);       { wait for entire init }
        none = pass ax0;            { buffer to be sent to }
        if ne jump check_init;      { the codec            }

check_aci1:
        idle;
        ay0 = 2;
        ax0 = dm (rx_buf);          { once initialized, wait for codec }
        ar = ax0 and ay0;           { to come out of autocalibration }
        if eq jump check_aci1;      { wait for bit set }


check_aci2:
        idle;
        ax0 = dm (rx_buf);          { wait for bit clear }
        ar = ax0 and ay0;
        if ne jump check_aci2;

        ay0 = 0xbf3f;               { unmute left DAC }
        ax0 = dm (init_cmds + 6);
        ar = ax0 AND ay0;
        dm (tx_buf) = ar;
        idle;

        ax0 = dm (init_cmds + 7);   { unmute right DAC }
        ar = ax0 AND ay0;
        dm (tx_buf) = ar;
        idle;


        ax0 = 0xa000;               { control word to clear over-range flags }
        dm (tx_buf) = ax0;
        idle;

        ifc = b#00000011111111;     { clear any pending interrupt }
        nop;

        imask = b#0000100000;       { enable rx0 interrupt }
            {     |||||||||+ | timer
                  ||||||||+- | SPORT1 rec or IRQ0
                  |||||||+-- | SPORT1 trx or IRQ1
                  ||||||+--- | BDMA
                  |||||+---- | IRQE
                  ||||+----- | SPORT0 rec
                  |||+------ | SPORT0 trx
                  ||+------- | IRQL0
                  |+-------- | IRQL1
                  +--------- | IRQ2
            }

{------------------------------------------------------------------------------
 -
 -  wait for interrupt and loop forever
 -
 ------------------------------------------------------------------------------}
        rts;






{******************************************************************************
 *
 *  Interrupt service routines
 *
 ******************************************************************************}

{------------------------------------------------------------------------------
 -
 -  receive interrupt used for loopback
 -
 -  REGISTER USAGE SUMMARY:
 -
 *  input  : dm (rx_buf)
 *  update : dm (codecBeep)
 *  output : dm (tx_buf)
 *  destroy: none (second register bank)
 *  keep   : none
 *  memory : none
 *  calls  : none
 -
 ------------------------------------------------------------------------------}
input_samples:
        ena sec_reg;                { use shadow register bank }

        ax1 = dm (rx_buf + 1);      { loopback inputs to outputs }
        mx1 = dm (rx_buf + 2);

        dm (tx_buf + 1) = ax1;
        dm (tx_buf + 2) = mx1;

{   puts out a tone if beep enabled }
        ar = dm (codecBeep);
        ar = ar - 1;
        if eq rti;
        dm (codecBeep) = ar;
        ay0 = %tone - 1;
        af = ar and ay0;
        ar = ^tone;
        ar = ar + af;
        ax0 = i7;
        i7 = ar;
        ar = dm (i7, m7);
        i7 = ax0;
        dm (tx_buf + 1) = ar;
        dm (tx_buf + 2) = ar;

        rti;


{------------------------------------------------------------------------------
 -
 -  transmit interrupt used for Codec initialization
 -
 -  REGISTER USAGE SUMMARY:
 -
 -  input  : m5 = 1
 -  update : i7
 -  output : tx0
 -  destroy: none (second register bank)
 -  keep   : none
 -  memory : dm (tx_buf), dm (stat_flg), init_cmds
 -  calls  : none
 -
 ------------------------------------------------------------------------------}
next_cmd:
        ena sec_reg;
        ax0 = dm (i7, m5);          { fetch next control word and }
        dm (tx_buf) = ax0;          { place in transmit slot 0    }
        ax0 = i7;
        ay0 = ^init_cmds;
        ar = ax0 - ay0;
        if gt rti;                  { rti if more control words still waiting }
        ax0 = 0xaf00;               { else set done flag and }
        dm (tx_buf) = ax0;          { remove MCE if done initialization }
        ax0 = 0;
        dm (stat_flag) = ax0;       { reset status flag }
        rti;



.endmod;
